#ifndef __TPointerCollection__
#define __TPointerCollection__

//	===========================================================================

#include "../Basics/CCountedObject.hpp"
#include "TPointerDeleter.hpp"
using Exponent::Basics::CCountedObject;
using Exponent::Collections::TPointerDeleter;
using Exponent::Collections::TObjectDeleter;
using Exponent::Collections::TObjectNuller;

//	===========================================================================


namespace Exponent
{
	namespace Collections
	{
		/**
		 * @class TPointerCollection TPointerCollection.hpp
		 * @brief Template'd pointer array that will grow and shrink as necessary
		 *
		 * Stores an internal array of pointer objects. Deletion is handled by the object deleter\n
		 * that you provide during construction. These templates allow you to handle deletion of\n
		 * objects in any way you require in a type safe way\n
		 * Because of the object management system, you are fully safe to do something like the following\n
		 * \n
		 * @code
		 * // Notice we add a new object directly, just like in java. The internal reference counting, depending on the pointer deleter
		 * // Will delete this directly
		 * array.addElement(new CString("Hello World"));
		 * @endcode
		 * \n
		 * *However* the object array does not expect array elements (ie you cant make a matrix) so therefore\n
		 * \n
		 * @code
		 * array.addElement(new double[32]);
		 * @endcode
		 * \n
		 * will not work without leaking memory like a bastard!\n
		 * The insert index is the current index the newly added objects will be assigned. It will always be <= m_arraySize\n
		 * To save time on loop, rather than doing\n
		 * \n
		 * @code
		 * for (long i = 0; i < array.getArraySize(); i++)
		 * @endcode
		 * \n
		 * You can do something like\n
		 * \n
		 * @code
		 * for (long i = 0; i < array.getInsertIndex(); i++)
		 * @endcode
		 * \n
		 * However, please mke sure that you check the validity of any pointer you get back, (as you should anyway!) as there\n
		 * is always the possibility of object positions being null. A good example is shown below\n
		 * \n
		 * @code
		 * TPointerCollection<CBool> boolArray;
		 * boolArray.addElement(new CBool(true));
		 * boolArray.addElement(new CBool(false));
		 * boolArray.addElement(new CBool(false));
		 *
		 * // Now we step through and print the results
		 * for (long i = 0; i < boolArray.getInsertIndex(); i++)
		 * {
		 *       // Get the pointer
		 *       CBool *boolean = boolArray.elementAtIndex(i);
		 *
		 *       // Check its valid
		 *       if (boolean == NULL) continue;
		 *
		 *       // Now you can process with the item
		 *       // ...
		 * }
		 * @endcode
		 *
		 * @see TPointerDeleter
		 *
		 * @date 23/01/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TPointerCollection.hpp,v 1.11 2007/02/08 21:06:44 paul Exp $
		 */
		template<class TypeName> class TPointerCollection : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			static TObjectDeleter<TypeName> TPOINTER_COLLECTION_DEFAULT_DELETER;		/**< The default handler for pointer deletion */
			static TObjectNuller<TypeName>  TPOINTER_COLLECTION_DEFAULT_NULLER;			/**< The defulat handler for nulling pointers */

//	===========================================================================

			const static long TPOINTERCOLLECTION_DEFAULT_GROW_SIZE	    = 32;			/**< Default amount to grow the array by */
			const static long TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER = -1;			/**< We failed to find a pointer */

//	===========================================================================

			/**
			 * @typedef qsortCompare
			 * @brief Comparison function pointer. This is an example of the implementation of a comparison function.\n
			 * To enable this to work properly with the qsort algorithm embedded inside the sort function you need to be capable of passing in double deferenced pointers\n
			 * This is a legal conversion from void *. A couple of words about the parameters:\n
			 * obj1 The first object to compare\n
			 * obj2 The second object\n
			 * You should return -1 if obj1 < obj2, 0 if obj1==obj2, +1 if obj1 > obj2\n
			 * To call this comparision using the default sorting of the pointer collection, do something like this:
			 * @code
			 * array.sortArray((TPointerCollection<CClass>::qsortCompare)CClass::compareClasses);
			 * @endcode
			 */
			typedef int (_cdecl *qsortCompare)(const void*, const void*);

//	===========================================================================

			/**
			 * Construction
			 * @param pointerDeleter The object that handles deletions
			 */
			TPointerCollection(TPointerDeleter<TypeName> *pointerDeleter = &TPOINTER_COLLECTION_DEFAULT_DELETER);

			/**
			 * Destruction
			 */
			virtual ~TPointerCollection();

//	===========================================================================

			/**
			 * Index operator
			 * @retval TypeName* The object or null on error
			 */
			TypeName *operator [] (const long index);

//	===========================================================================

			/**
			 * Get the element at a specific index
			 * @param index The index of the pointer
			 * @retval TypeName* The object requested or NULL on error
			 */
			TypeName *elementAtIndex(const long index);

			/**
			 * Get the element at a specific index
			 * @param index The index of the pointer
			 * @retval TypeName* The object requested or NULL on error
			 */
			const TypeName *constElementAtIndex(const long index) const;

			/**
			 * Add a pointer to the array
			 * @param pointer The pointer to add
			 */
			void addElement(TypeName *pointer);

			/**
			 * Insert a pointer at a specific index
			 * @param index The index to store at, if an object is there it is deleted
			 * @param pointer The pointer to add
			 */
			void addElementAtIndex(const long index, TypeName *pointer);

			/**
			 * Delete a pointer at an index
			 * @param index Index of the pointer to delete
			 */
			void deletePointerAtIndex(const long index);

			/**
			 * Remove the pointer at the index without deletion
			 * @param index The index of the pointer to remove
			 * @retval TypeName* The object removed from the array
			 * @note This NEVER, EVER deletes the object, it simply nulls its position, you become responsible for deletion
			 */
			TypeName *removeElementAtIndexWithoutDeletion(const long index);

//	===========================================================================

			/**
			 * Swap two pointers
			 * @param index1 The index of the first pointer
			 * @param index2 The index of the second pointer
			 */
			void swapIndexes(const long index1, const long index2);

			/**
			 * Reorder the array to remove dead pointers, resize shrinks the array
			 * @param resize If true will remove any remaining null pointers at the end of the array, if false array may be bigger than usage
			 */
			void reorder(const bool resize = true);

//	===========================================================================

			/**
			 * Get the index of a pointer
			 * @param pointer The pointer to find
			 * @retval long The index of the pointer or TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER on error
			 */
			long getIndexOfPointer(const TypeName *pointer);

			/**
			 * Is pointer in array
			 * @param pointer The object to look for
			 * @retval bool True if this object is stored in the array, false otherwise
			 */
			bool isPointerInArray(const TypeName *pointer);

			/**
			 * Get the size of the array
			 * @retval long The size of the array
			 */
			long getArraySize() const;

			/**
			 * Get the insert index
			 * @retval long The current end pointer of the array
			 */
			long getInsertIndex() const;

			/**
			 * Clear the array
			 */
			void clearArray();

			/**
			 * Is the array empty?
			 * @retval bool True if array size is zero
			 */
			bool isArrayEmpty() const;

			/**
			 * Set the grow size
			 * @param growSize The amount that the array size should be increased by
			 */
			void setGrowSize(const long growSize = TPOINTERCOLLECTION_DEFAULT_GROW_SIZE);

			/**
			 * Get number of elements until next grow
			 * @retval long The number of elements that can be added before the array has to grow
			 */
			long getNumberOfElementsUntilNextGrow();

//	===========================================================================

			/**
			 * Set the pointer deleter
			 * @param pointerDeleter The object that handles deletions
			 */
			void registerPointerDeleter(TPointerDeleter<TypeName> *pointerDeleter = &TPOINTER_COLLECTION_DEFAULT_DELETER) { m_pointerDeletionHandler = pointerDeleter; }

//	===========================================================================

			/**
			 * Get the internal buffer
			 * @retval const TypeName ** The internal buffer
			 */
			TypeName **getMutableInternalBuffer() { return m_array; }

			/**
			 * Sort the array
			 * @param compareFunction The comparison function to use
			 */
			void sortArray(qsortCompare compareFunction);

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Initialise the unit
			 */
			void initialise();

			/**
			 * Expand the array
			 */
			void expand();

			/**
			 * Free pointers -> deleteds (but not allways) all pointers
			 */
			void freePointers();

//	===========================================================================

			TypeName **m_array;											/**< The internal array */
			TPointerDeleter<TypeName> *m_pointerDeletionHandler;		/**< Deletion handler */
			long m_arraySize;											/**< The size of the array */
			long m_growSize;											/**< how many extra area do we require when we run out */
			long m_insertIndex;											/**< The current end point of the array < m_arraySize */

//	===========================================================================

		};

		/**
		 * @cond
		 */

		 EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TPointerCollection<TypeName>, TypeName, CCountedObject);

		 template<class TypeName> TObjectDeleter<TypeName> TPointerCollection<TypeName>::TPOINTER_COLLECTION_DEFAULT_DELETER;
		 template<class TypeName> TObjectNuller<TypeName>  TPointerCollection<TypeName>::TPOINTER_COLLECTION_DEFAULT_NULLER;

//	===========================================================================
		 template<class TypeName> TPointerCollection<TypeName>::TPointerCollection(TPointerDeleter<TypeName> *pointerDeleter)
															  : m_array(NULL)
															  , m_pointerDeletionHandler(NULL)
															  , m_arraySize(0)
															  , m_growSize(0)
															  , m_insertIndex(0)
		{
			EXPONENT_CLASS_CONSTRUCTION(TPointerCollection<TypeName>);

			// First we want to register the pointer deleter
			this->registerPointerDeleter(pointerDeleter);

			// Set the default growing size
			this->setGrowSize();

			// We have no pointers just yet
			m_arraySize   = 0;
			m_insertIndex = 0;
			NULL_POINTER(m_array);
		}

//	===========================================================================
		template<class TypeName> TPointerCollection<TypeName>::~TPointerCollection()
		{
			EXPONENT_CLASS_DESTRUCTION(TPointerCollection<TypeName>);
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> TypeName *TPointerCollection<TypeName>::operator [] (const long index)
		{
			return this->elementAtIndex(index);
		}
//	===========================================================================
		template<class TypeName> TypeName *TPointerCollection<TypeName>::elementAtIndex(const long index)
		{
			// Check the index is within the correct range and we have valid pointers
			if (m_array && (index >= 0) && (index < m_arraySize))
			{
				// Return the pointer
				return m_array[index];
			}

			// They failed
			return NULL;
		}
//	===========================================================================
		template<class TypeName> const TypeName *TPointerCollection<TypeName>::constElementAtIndex(const long index) const
		{
			// Check the index is within the correct range and we have valid pointers
			if (m_array && (index >= 0) && (index < m_arraySize))
			{
				// Return the pointer
				return m_array[index];
			}

			// They failed
			return NULL;
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::addElement(TypeName *pointer)
		{
			// If we dont yet have an array, we want to create it
			if (m_array == NULL)
			{
				this->initialise();
			}

			// Check if we need more elements
			if (m_insertIndex >= m_arraySize)
			{
				this->expand();
			}

			// Add the element to the array
			m_array[m_insertIndex++] = pointer;

			// Notify handler
			if (m_pointerDeletionHandler)
			{
				m_pointerDeletionHandler->pointerAdded(pointer);
			}
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::addElementAtIndex(const long index, TypeName *pointer)
		{
			// Check we have valid indicies and pointers
			if (m_array && (index >= 0 && index < m_arraySize))
			{
				// Delete the current position
				this->deletePointerAtIndex(index);

				// Now null the pointer in the array
				m_array[index] = pointer;

				// Notify handler
				if (m_pointerDeletionHandler)
				{
					m_pointerDeletionHandler->pointerAdded(pointer);
				}
			}
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::deletePointerAtIndex(const long index)
		{
			// Check we have valid indicies and pointers
			if (m_array && (index >= 0 && index < m_arraySize))
			{
				// If we have a handler, let it do the deleting
				if (m_pointerDeletionHandler)
				{
					// Check its valid
					if (m_array[index] != NULL)
					{
						// Delete it
						m_pointerDeletionHandler->deletePointer(m_array[index]);
					}
				}

				// Now null the pointer in the array
				NULL_POINTER(m_array[index]);
			}
		}

//	===========================================================================
		template<class TypeName> TypeName *TPointerCollection<TypeName>::removeElementAtIndexWithoutDeletion(const long index)
		{
			// Check we have pointers and valid indicies
			if (m_array && (index >= 0 && index < m_arraySize))
			{
				// Get the object
				TypeName *object = m_array[index];

				// Null the position pointer
				NULL_POINTER(m_array[index]);

				// Return the new object
				return object;
			}

			// They failed!
			return NULL;
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::swapIndexes(const long index1, const long index2)
		{
			// If we have validity
			if (m_array && index1 >= 0 && index2 >= 0 && index1 < m_arraySize && index2 < m_arraySize)
			{
				// Copy the pointer
				TypeName *ptr   = m_array[index1];

				// Move first pointer
				m_array[index1] = m_array[index2];

				// STore the copy
				m_array[index2] = ptr;
			}
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::reorder(const bool resize)
		{
			// Check that we have values that are valid
			if (m_array == NULL)
			{
				return;
			}

			// Create a temporary array
			TypeName **tempArray = new TypeName*[m_arraySize];

			// Pointr to the final size
			long index = 0;

			// Copy everything
			for (long i = 0; i < m_arraySize; i++)
			{
				// Null the position
				NULL_POINTER(tempArray[i]);

				// If its non null
				if (m_array[i] != NULL)
				{
					// We want to add it
					tempArray[index++] = m_array[i];
				}
			}

			// We only resize if requested
			if (resize)
			{
				// We want to shrink the array
				if (index == m_arraySize)
				{
					// Dont need to do anything, no nulls
				}
				else
				{
					// Fewer elements
					TypeName **tempArray2 = new TypeName*[index];

					// Copy everything
					for (long i = 0; i < index; i++)
					{
						// We want to store it
						tempArray2[i] = tempArray[i];
					}

					// Store the new size
					m_arraySize   = index;
					m_insertIndex = m_arraySize;

					// Delete the old array
					FREE_ARRAY_POINTER(m_array);
					FREE_ARRAY_POINTER(tempArray);

					// Store the array
					m_array = tempArray2;

				}
			}
			else
			{
				// Delete the old array
				FREE_ARRAY_POINTER(m_array);

				// Store the array
				m_array = tempArray;

				// Store the size ad the insert index
				// Size of the array has not changed
				m_insertIndex = index;
			}
		}

//	===========================================================================
		template<class TypeName> long TPointerCollection<TypeName>::getIndexOfPointer(const TypeName *pointer)
		{
			// If we have an array
			if (m_array)
			{
				// Loop through the pointers
				for (long i = 0; i < m_arraySize; i++)
				{
					// If its the one..
					if (pointer == m_array[i])
					{
						// Then we are done
						return i;
					}
				}
			}

			// Pointer is not in the array
			return TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER;
		}

//	===========================================================================
		template<class TypeName> bool TPointerCollection<TypeName>::isPointerInArray(const TypeName *pointer)
		{
			return (this->getIndexOfPointer(pointer) != TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER);
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::clearArray()
		{
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> long TPointerCollection<TypeName>::getArraySize() const
		{
			return m_arraySize;
		}

//	===========================================================================
		template<class TypeName> long TPointerCollection<TypeName>::getInsertIndex() const
		{
			return m_insertIndex;
		}

//	===========================================================================
		template<class TypeName> bool TPointerCollection<TypeName>::isArrayEmpty() const
		{
			return (m_arraySize == 0);
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::setGrowSize(const long growSize)
		{
			m_growSize = growSize;
		}

//	===========================================================================
		template<class TypeName> long TPointerCollection<TypeName>::getNumberOfElementsUntilNextGrow()
		{
			return m_arraySize - m_insertIndex;
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::sortArray(qsortCompare compareFunction)
		{
			qsort(m_array, m_arraySize, sizeof(TypeName *), compareFunction);
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::initialise()
		{
			// Delete any old array that we may have
			this->freePointers();

			// Expand to the initial size
			this->expand();
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::expand()
		{
			// Create a temporary array
			TypeName **tempArray = new TypeName*[m_arraySize + m_growSize];

			// Copy everything in
			for (long i = 0; i < m_arraySize + m_growSize; i++)
			{
				// Copy as necessary
				tempArray[i] = (i < m_arraySize) ? m_array[i] : NULL;
			}

			// Increase the array size
			m_arraySize += m_growSize;

			// Delete the old array
			FREE_ARRAY_POINTER(m_array);

			// Now store the new pointers
			m_array = tempArray;
		}

//	===========================================================================
		template<class TypeName> void TPointerCollection<TypeName>::freePointers()
		{
			// Only if we have pointers
			if (m_array)
			{
				// If we have a handler, let it do the deleting
				if (m_pointerDeletionHandler)
				{
					// Delete all the elements
					for (long i = 0; i < m_arraySize; i++)
					{
						// Check its valid
						if (m_array[i] != NULL)
						{
							// Delete it
							m_pointerDeletionHandler->deletePointer(m_array[i]);
						}
					}
				}

				// Delete the array
				FREE_ARRAY_POINTER(m_array);
			}
			m_arraySize   = 0;
			m_insertIndex = 0;
		}

		/** @endcond */
	}
}
#endif	// End of TPointerCollection.hpp